/**
  LoRaWAN Irrigation System using the Arduino Edge Control - Application Note
  Name: Edge_Control_Code.ino
  Purpose: 4 Zones Smart Irrigation System using the Arduino Edge Control with Cloud connectivity using a MKR WAN 1310.

  @author Christopher Mendez
*/

#include <Arduino_EdgeControl.h>
#include <Wire.h>
#include <RunningMedian.h>

#include "SensorValues.hpp"
#include "Helpers.h"

// The MKR1 board I2C address
#define EDGE_I2C_ADDR 0x05

constexpr unsigned int adcResolution{ 12 };  // Analog Digital Converter resolution for the Watermark sensors.

mbed::LowPowerTimeout TimerM;

// Watermark sensors thresholds
const long open_resistance = 35000, short_resistance = 200, short_CB = 240, open_CB = 255, TempC = 28;

// Watermark sensors channels
uint8_t watermarkChannel[4] = { 0, 1, 2, 3 };

constexpr float tauRatio{ 0.63f };
constexpr float tauRatioSamples{ tauRatio * float{ (1 << adcResolution) - 1 } };
constexpr unsigned long sensorDischargeDelay{ 2 };

constexpr unsigned int measuresCount{ 20 };
RunningMedian measures{ measuresCount };

constexpr unsigned int calibsCount{ 10 };
RunningMedian calibs{ calibsCount };

unsigned long previousMillis = 0;  // will store last time the sensors were updated

const long interval = 180000;  // interval of the LoRaWAN message (milliseconds)

// Variables for the water flow measurement
volatile int irqCounts;
float calibrationFactor = 4.5;
volatile byte pulseCount = 0;
float flowRate = 0.0;
unsigned int flowMilliLitres = 0;
unsigned long totalMilliLitres = 0;
unsigned long oldTime = 0;
unsigned long oldTime2 = 0;

// Valves flow control variables
bool controlV1 = 1;
bool controlV2 = 1;
bool controlV3 = 1;
bool controlV4 = 1;

// Valves On time keeping variables
int StartTime1, CurrentTime1;
int StartTime2, CurrentTime2;
int StartTime3, CurrentTime3;
int StartTime4, CurrentTime4;

// LCD flow control variables
bool controlLCD = 1;
int showTimeLCD = 0;

// Smart mode variables
#define dry_soil 30
bool smart = false;
bool V1open = 0;
bool V2open = 0;
bool V3open = 0;
bool V4open = 0;
/** UI Management **/
// Button statuses
enum ButtonStatus : byte {
  ZERO_TAP,
  SINGLE_TAP,
  DOUBLE_TAP,
  TRIPLE_TAP,
  QUAD_TAP,
  FIVE_TAP,
  LOT_OF_TAPS
};

// ISR: count the button taps
volatile byte taps{ 0 };
// ISR: keep elapsed timings
volatile unsigned long previousPress{ 0 };
// ISR: Final button status
volatile ButtonStatus buttonStatus{ ZERO_TAP };

SensorValues_t vals;

/**
  Main section setup
*/
void setup() {

  EdgeControl.begin();
  Wire.begin();

  delay(500);
  Serial.begin(115200);
  delay(2000);

  Power.enable3V3();
  Power.enable5V();
  Power.on(PWR_3V3);
  Power.on(PWR_VBAT);
  Power.on(PWR_MKR1);

  delay(5000);  // giving time for the MKR WAN 1310 to boot

  // Init Edge Control IO Expander
  Serial.print("IO Expander initializazion ");
  if (!Expander.begin()) {
    Serial.println("failed.");
    Serial.println("Please, be sure to enable gated 3V3 and 5V power rails");
    Serial.println("via Power.enable3V3() and Power.enable5V().");
  } else Serial.println("succeeded.");

  // Init IRQ INPUT pins
  pinMode(IRQ_CH1, INPUT);

  // Attach callbacks to IRQ pins
  attachInterrupt(digitalPinToInterrupt(IRQ_CH1), [] {irqCounts++;},FALLING);

  // LCD button definition
  pinMode(POWER_ON, INPUT);
  attachInterrupt(POWER_ON, buttonPress, RISING);

  Watermark.begin();
  Latching.begin();
  analogReadResolution(adcResolution);

  setSystemClock(__DATE__, __TIME__);  // define system time as a reference for the RTC

  // Init the LCD display
  LCD.begin(16, 2);
  LCD.backlight();

  LCD.home();
  LCD.print("LoRa Irrigation");
  LCD.setCursor(5, 1);
  LCD.print("System");

  CloseAll();

  delay(2000);  // closing all the valves.

  LCD.clear();
}

/**
 Main section loop
*/
void loop() {

  // LCD button taps detector function
  detectTaps();
  tapsHandler();

  // reset the valves accumuldated on time every day at midnight
  if (getLocalhour() == " 00:00:00") {
    Serial.println("Resetting accumulators every day");
    vals.z1_on_time_local = 0;
    vals.z2_on_time_local = 0;
    vals.z3_on_time_local = 0;
    vals.z4_on_time_local = 0;
    delay(1000);
  }

  readWatermark();

  if ((millis() - oldTime2) >= 1000)  // Only process counters once per second
  {
    oldTime2 = millis();
    readWaterFLow();
    auto vbat = Power.getVBat(adcResolution);
    Serial.print("Battery Voltage: ");
    Serial.println(vbat);
    vals.battery_volt_local = vbat;
  }

  unsigned long currentMillis = millis();

  if (currentMillis - previousMillis >= interval) {

    previousMillis = currentMillis;

    // send local sensors values and retrieve cloud variables status back and forth
    Serial.println("Sending variables to MKR");
    updateSensors();
  }

  // activate, deactivate and keep time of valves function
  valvesHandler();
}

/**
  This function sends the variables to the MKR through I2C to be sent to the Cloud
*/
void updateSensors() {
  sendValues(&vals);
}

/**
  This function calculates the estimated water used by the system measuring the interruption
  frequency triggered by the water flow sensor.
*/
void readWaterFLow() {
  // Disable the interrupt while calculating flow rate and sending the value to
  // the host
  detachInterrupt(digitalPinToInterrupt(IRQ_CH1));

  // Because this loop may not complete in exactly 1 second intervals we calculate
  // the number of milliseconds that have passed since the last execution and use
  // that to scale the output. We also apply the calibrationFactor to scale the output
  // based on the number of pulses per second per units of measure (litres/minute in
  // this case) coming from the sensor.
  flowRate = ((1000.0 / (millis() - oldTime)) * irqCounts) / calibrationFactor;

  // Note the time this processing pass was executed. Note that because we've
  // disabled interrupts the millis() function won't actually be incrementing right
  // at this point, but it will still return the value it was set to just before
  // interrupts went away.
  oldTime = millis();

  // Divide the flow rate in litres/minute by 60 to determine how many litres have
  // passed through the sensor in this 1 second interval, then multiply by 1000 to
  // convert to millilitres.
  flowMilliLitres = (flowRate / 60) * 1000;

  // Add the millilitres passed in this second to the cumulative total
  totalMilliLitres += flowMilliLitres;

  unsigned int frac;

  // Print the flow rate for this second in litres / minute
  Serial.print("Flow rate: ");
  Serial.print(int(flowRate));  // Print the integer part of the variable
  Serial.print("L/min");
  Serial.print("\t");  // Print tab space

  // Print the cumulative total of litres flowed since starting
  Serial.print("Output Liquid Quantity: ");
  Serial.print(totalMilliLitres);
  Serial.print("mL");
  Serial.print("\t");  // Print tab space
  Serial.print(totalMilliLitres / 1000);
  Serial.println("L");
  vals.water_usage_local = totalMilliLitres / 1000;
  vals.water_flow_local = int(flowRate);

  // Reset the pulse counter so we can start incrementing again
  irqCounts = 0;

  // Enable the interrupt again now that we've finished sending output
  attachInterrupt(digitalPinToInterrupt(IRQ_CH1), [] {irqCounts++;},FALLING);
}

/**
  This function reads the watermark sensors using the Tau method.
*/
void readWatermark() {
  static bool highPrec{ true };
  Watermark.setHighPrecision(highPrec);
  //highPrec = !highPrec;

  // Init commands and reset devices
  Watermark.calibrationMode(OUTPUT);
  Watermark.calibrationWrite(LOW);
  Watermark.commonMode(OUTPUT);
  Watermark.commonWrite(LOW);

  Watermark.fastDischarge(sensorDischargeDelay);

  // Calibration cycle:
  // disable Watermark demuxer
  for (int j = 0; j < 4; j++) {
    WatermarkCal(watermarkChannel[j]);
    Watermark.fastDischarge(sensorDischargeDelay);
  }

  // Measures cycle:
  // enable Watermark demuxer

  for (int i = 0; i < 4; i++) {
    WatermarkGet(watermarkChannel[i]);
    Serial.print("Channel 0" + String(i + 1) + " ");
    Serial.print("- Average Resistance: ");
    Serial.print(measures.getAverage());
    Serial.print(" - Average CB: ");
    int CBreading = CalcCB(measures.getAverage());
    Serial.print(CBreading);
    switch (i) {
      case 0:
        vals.z1_moisture_local = CBreading;
        break;
      case 1:
        vals.z2_moisture_local = CBreading;
        break;
      case 2:
        vals.z3_moisture_local = CBreading;
        break;
      case 3:
        vals.z4_moisture_local = CBreading;
        break;
      default:
        Serial.println("Should never execute");
    }
    measures.clear();
    Serial.println();
  }
  Serial.println();
}

/**
  This function handles the specific watermark channel measurement
  @param WK_ch defines the channel to sample
*/
void WatermarkGet(uint8_t WK_ch) {

  Watermark.enable();

  Watermark.commonMode(OUTPUT);
  Watermark.calibrationMode(INPUT);
  for (auto i = 0u; i < measuresCount; i++) {
    Watermark.commonWrite(HIGH);
    auto start = micros();
    while (Watermark.analogRead(WK_ch) < tauRatioSamples)
      ;
    auto stop = micros();
    Watermark.commonWrite(LOW);
    Watermark.fastDischarge(sensorDischargeDelay);
    measures.add(stop - start);
  }
}

/**
  This function handles the watermark calibration
  @param WK_ch defines the channel to calibrate
*/
void WatermarkCal(uint8_t WK_ch) {

  // Calibration cycle:
  // disable Watermark demuxer
  Watermark.disable();

  Watermark.commonMode(INPUT);
  Watermark.calibrationMode(OUTPUT);
  for (auto i = 0u; i < measuresCount; i++) {
    Watermark.calibrationWrite(HIGH);

    auto start = micros();
    while (Watermark.analogRead(WK_ch) < tauRatioSamples)
      ;
    auto stop = micros();
    Watermark.calibrationWrite(LOW);
    Watermark.fastDischarge(sensorDischargeDelay);
    calibs.add(stop - start);
  }
}

/**
  This function convert the Watermark readings into centibars
  @param res is the resistance measured of the watermark sensor
  @return CB, the centibars
*/
int CalcCB(int res) {
  int CB = 0;
  if (res > 550.00) {

    if (res > 8000.00) {
      CB = -2.246 - 5.239 * (res / 1000.00) * (1 + .018 * (TempC - 24.00)) - .06756 * (res / 1000.00) * (res / 1000.00) * ((1.00 + 0.018 * (TempC - 24.00)) * (1.00 + 0.018 * (TempC - 24.00)));
    } else if (res > 1000.00) {
      CB = (-3.213 * (res / 1000.00) - 4.093) / (1 - 0.009733 * (res / 1000.00) - 0.01205 * (TempC));
    } else {
      CB = ((res / 1000.00) * 23.156 - 12.736) * (1.00 + 0.018 * (TempC - 24.00));
    }
  } else {
    if (res > 300.00) {
      CB = 0.00;
    }
    if (res < 300.00 && res >= short_resistance) {
      CB = short_CB;  //240 is a fault code for sensor terminal short
    }
  }

  if (res >= open_resistance) {
    CB = open_CB;  //255 is a fault code for open circuit or sensor not present
  }

  return abs(CB);
}

/**
  Poor-man LCD button debouncing function. 
*/
void buttonPress() {
  const auto now = millis();

  if (now - previousPress > 100)
    taps++;

  previousPress = now;
}

/**
  Detect and count button taps
*/
void detectTaps() {
  // Timeout to validate the button taps counter
  constexpr unsigned int buttonTapsTimeout{ 300 };

  // Set the button status and reset the taps counter when button has been
  // pressed at least once and button taps validation timeout has been reached.
  if (taps > 0 && millis() - previousPress >= buttonTapsTimeout) {
    buttonStatus = static_cast<ButtonStatus>(taps);
    taps = 0;
  }
}

/**
  Execute the right action between different taps
*/
void tapsHandler() {
  // Different button taps handler
  switch (buttonStatus) {
    case ZERO_TAP:  // will execute always the button is not being pressed.
      if (controlLCD == 1) {
        // you can put something here that executes just once.
        controlLCD = 0;
      }
      ZonesMoistureLCD();  // while the valves are not being controlled
      delay(1000);
      break;

    case SINGLE_TAP:  // will execute when the button is pressed once.
      Serial.println("Single Tap");
      vals.valve1_local = !vals.valve1_local;
      sendValues(&vals);
      buttonStatus = ZERO_TAP;
      break;

    case DOUBLE_TAP:  // will execute when the button is pressed twice.
      Serial.println("Double Tap");
      vals.valve2_local = !vals.valve2_local;
      sendValues(&vals);
      buttonStatus = ZERO_TAP;
      break;

    case TRIPLE_TAP:  // will execute when the button is pressed three times.
      Serial.println("Triple Tap");
      vals.valve3_local = !vals.valve3_local;
      sendValues(&vals);
      buttonStatus = ZERO_TAP;
      break;

    case QUAD_TAP:  // will execute when the button is pressed four times.
      Serial.println("Quad Tap");
      vals.valve4_local = !vals.valve4_local;
      sendValues(&vals);
      buttonStatus = ZERO_TAP;
      break;

    case FIVE_TAP:  // will execute when the button is pressed five times.
      Serial.println("Five Tap");
      smart = !smart;
      Serial.print("Smart mode ");
      if (smart == true) {
        Serial.println("enabled.");
        LCD.clear();
        LCD.setCursor(0, 0);
        LCD.print("Smart Irrigation");
        LCD.setCursor(4, 1);
        LCD.print("Enabled");
      } else {
        Serial.println("disabled.");
        LCD.clear();
        LCD.setCursor(0, 0);
        LCD.print("Smart Irrigation");
        LCD.setCursor(4, 1);
        LCD.print("Disabled");
      }
      buttonStatus = ZERO_TAP;
      break;

    default:
      Serial.println("Too Many Taps");
      buttonStatus = ZERO_TAP;
      break;
  }
}

/**
  Function that sends the local sensors values through I2C to the MKR
  @param values The I2C communicated sensors values
*/
void sendValues(SensorValues_t *values) {
  writeBytes((uint8_t *)values, sizeof(SensorValues_t));
}

/**
  Function that transport the sensors data through I2C to the MKR
  @param buf store the structured sensors values
  @param len store the buffer lenght
*/
void writeBytes(uint8_t *buf, uint8_t len) {

  Wire.beginTransmission(EDGE_I2C_ADDR);

  for (uint8_t i = 0; i < len; i++) {
    Wire.write(buf[i]);
  }

  Wire.endTransmission();
}

/**
  Function that controls the solenoid valves and
  measures the ON time of each one and update their status on the LCD screen.
*/
void valvesHandler() {

  if (smart == true) {  // smart mode irrigates based on soil moisture

    if (vals.z1_moisture_local >= dry_soil && V1open == 0) {  // opening valve 1
      Latching.channelDirection(LATCHING_OUT_1, POSITIVE);
      Latching.strobe(30);  // Do not increase the pulse lenght more than this.
      V1open = 1;
    } else if (vals.z1_moisture_local < dry_soil && V1open == 1) {
      Latching.channelDirection(LATCHING_OUT_1, NEGATIVE);
      Latching.strobe(30);  // Do not increase the pulse lenght more than this.
      V1open = 0;
    }

    if (vals.z2_moisture_local >= dry_soil && V2open == 0) {  // opening valve 2
      Latching.channelDirection(LATCHING_OUT_3, POSITIVE);
      Latching.strobe(30);  // Do not increase the pulse lenght more than this.
      V2open = 1;
    } else if (vals.z2_moisture_local < dry_soil && V2open == 1) {
      Latching.channelDirection(LATCHING_OUT_3, NEGATIVE);
      Latching.strobe(30);  // Do not increase the pulse lenght more than this.
      V2open = 0;
    }

    if (vals.z3_moisture_local >= dry_soil && V3open == 0) {  // opening valve 3
      Latching.channelDirection(LATCHING_OUT_5, POSITIVE);
      Latching.strobe(30);  // Do not increase the pulse lenght more than this.
      V3open = 1;
    } else if (vals.z3_moisture_local < dry_soil && V3open == 1) {
      Latching.channelDirection(LATCHING_OUT_5, NEGATIVE);
      Latching.strobe(30);  // Do not increase the pulse lenght more than this.
      V3open = 0;
    }

    if (vals.z4_moisture_local >= dry_soil && V4open == 0) {  // opening valve 4
      Latching.channelDirection(LATCHING_OUT_7, POSITIVE);
      Latching.strobe(30);  // Do not increase the pulse lenght more than this.
      V4open = 1;
    } else if (vals.z4_moisture_local < dry_soil && V4open == 1) {
      Latching.channelDirection(LATCHING_OUT_7, NEGATIVE);
      Latching.strobe(30);  // Do not increase the pulse lenght more than this.
      V4open = 0;
    }

  } else {
    if (vals.valve1_local == 1 && controlV1 == 1) {
      Serial.println("Opening Valve 1");
      LCD.clear();
      LCD.setCursor(1, 0);
      LCD.print("Opening Valve");
      LCD.setCursor(7, 1);
      LCD.print("#1");
      controlLCD = 1;
      Latching.channelDirection(LATCHING_OUT_1, POSITIVE);
      Latching.strobe(30);  // Do not increase the pulse lenght more than this.
      StartTime1 = time(NULL);
      controlV1 = 0;
      delay(2000);
      LCD.clear();
    } else if (vals.valve1_local == 0 && controlV1 == 0) {
      Serial.println("Closing Valve 1");
      LCD.clear();
      LCD.setCursor(1, 0);
      LCD.print("Closing Valve");
      LCD.setCursor(7, 1);
      LCD.print("#1");
      controlLCD = 1;
      Latching.channelDirection(LATCHING_OUT_1, NEGATIVE);
      Latching.strobe(30);  // Do not increase the pulse lenght more than this.
      CurrentTime1 = time(NULL);
      Serial.print("V1 On Time: ");
      vals.z1_on_time_local += (CurrentTime1 - StartTime1) / 60.0;
      Serial.println(vals.z1_on_time_local);
      controlV1 = 1;
      delay(2000);
      LCD.clear();
    }

    if (vals.valve2_local == 1 && controlV2 == 1) {
      Serial.println("Opening Valve 2");
      LCD.clear();
      LCD.setCursor(1, 0);
      LCD.print("Opening Valve");
      LCD.setCursor(7, 1);
      LCD.print("#2");
      controlLCD = 1;
      Latching.channelDirection(LATCHING_OUT_3, POSITIVE);
      Latching.strobe(30);  // Do not increase the pulse lenght more than this.
      StartTime2 = time(NULL);
      controlV2 = 0;
      delay(2000);
      LCD.clear();
    } else if (vals.valve2_local == 0 && controlV2 == 0) {
      Serial.println("Closing Valve 2");
      LCD.clear();
      LCD.setCursor(1, 0);
      LCD.print("Closing Valve");
      LCD.setCursor(7, 1);
      LCD.print("#2");
      controlLCD = 1;
      Latching.channelDirection(LATCHING_OUT_3, NEGATIVE);
      Latching.strobe(30);  // Do not increase the pulse lenght more than this.
      CurrentTime2 = time(NULL);
      Serial.print("V2 On Time: ");
      vals.z2_on_time_local += (CurrentTime2 - StartTime2) / 60.0;
      Serial.println(vals.z2_on_time_local);
      controlV2 = 1;
      delay(2000);
      LCD.clear();
    }

    if (vals.valve3_local == 1 && controlV3 == 1) {
      Serial.println("Opening Valve 3");
      LCD.clear();
      LCD.setCursor(1, 0);
      LCD.print("Opening Valve");
      LCD.setCursor(7, 1);
      LCD.print("#3");
      controlLCD = 1;
      Latching.channelDirection(LATCHING_OUT_5, POSITIVE);
      Latching.strobe(30);   // Do not increase the pulse lenght more than this.
      StartTime3 = time(NULL);
      controlV3 = 0;
      delay(2000);
      LCD.clear();
    } else if (vals.valve3_local == 0 && controlV3 == 0) {
      Serial.println("Closing Valve 3");
      LCD.clear();
      LCD.setCursor(1, 0);
      LCD.print("Closing Valve");
      LCD.setCursor(7, 1);
      LCD.print("#3");
      controlLCD = 1;
      Latching.channelDirection(LATCHING_OUT_5, NEGATIVE);
      Latching.strobe(30);  // Do not increase the pulse lenght more than this.
      CurrentTime3 = time(NULL);
      Serial.print("V3 On Time: ");
      vals.z3_on_time_local += (CurrentTime3 - StartTime3) / 60.0;
      Serial.println(vals.z3_on_time_local);
      controlV3 = 1;
      delay(2000);
      LCD.clear();
    }

    if (vals.valve4_local == 1 && controlV4 == 1) {
      Serial.println("Opening Valve 4");
      LCD.clear();
      LCD.setCursor(1, 0);
      LCD.print("Opening Valve");
      LCD.setCursor(7, 1);
      LCD.print("#4");
      controlLCD = 1;
      Latching.channelDirection(LATCHING_OUT_7, POSITIVE);
      Latching.strobe(30);  // Do not increase the pulse lenght more than this.
      StartTime4 = time(NULL);
      controlV4 = 0;
      delay(2000);
      LCD.clear();
    } else if (vals.valve4_local == 0 && controlV4 == 0) {
      Serial.println("Closing Valve 4");
      LCD.clear();
      LCD.setCursor(1, 0);
      LCD.print("Closing Valve");
      LCD.setCursor(7, 1);
      LCD.print("#4");
      controlLCD = 1;
      Latching.channelDirection(LATCHING_OUT_7, NEGATIVE);
      Latching.strobe(30);  // Do not increase the pulse lenght more than this.
      CurrentTime4 = time(NULL);
      Serial.print("V4 On Time: ");
      vals.z4_on_time_local += (CurrentTime4 - StartTime4) / 60.0;
      Serial.println(vals.z4_on_time_local);
      controlV4 = 1;
      delay(2000);
      LCD.clear();
    }
  }
}

/**
  Close all the valves
*/
void CloseAll() {
  for (int i = 1; i <= 7; i++) {
    Latching.channelDirection(i, NEGATIVE);
    Latching.strobe(30);  // Do not increase the pulse lenght more than this.
    delay(500);
  }
}

/**
  Function that shows each valve current countdown timer on the LCD screen.
*/
void ZonesMoistureLCD() {

  char line1[32];
  char line2[32];

  //LCD.clear();

  sprintf(line1, "Z1:%dCB Z2:%dCB    ", vals.z1_moisture_local, vals.z2_moisture_local);  // use sprintf() to compose the string line1
  sprintf(line2, "Z3:%dCB Z4:%dCB    ", vals.z3_moisture_local, vals.z4_moisture_local);  // use sprintf() to compose the string line2

  LCD.setCursor(0, 0);
  LCD.print(line1);
  LCD.setCursor(0, 1);
  LCD.print(line2);
}
